' Rudenko's Disk Puzzle
' Rev 1.0.0 William M Leue 16-Dec-2023

option default integer
option base 1
option angle degrees

' Constants

' Geometry
const CASE_RAD         = 275
const OUTER_RAD        = 215
const INNER_RAD        = 110
const TRACK_WIDTH      = 25
const OUTER_TRACK_RAD  = OUTER_RAD-TRACK_WIDTH\2
const INNER_TRACK_RAD  = INNER_RAD-TRACK_WIDTH\2
const CX = mm.hres\2
const CY = mm.vres\2
const DISK_START_ANGLE = 117
const DISK_ANGLE_INC   = 21
const DISK_OUTER_RAD   = 34
const DISK_INNER_RAD   = 23
const NUM_DISKS        = 7
const NUM_SLOTS        = 3*NUM_DISKS

' sliding track polygon
const NVERTICES = 219

' keyboard commands
const ESC   = 27
const HOME  = 134
const ATK   = 64
const PGDWN = 137
const F1    = 145
const QUEST = 63
const PLUS  = 43
const MINUS = 45

' track branch codes
' (the branch point is not itself a branch)
const BRA_LO = 1
const BRA_RO = 2
const BRA_IN = 3
const NBRA  = 3

' disk component indices
const DSK_SLOT     = 1
const DSK_INDEX    = 2
const DSK_SELECTED = 3

' Desaturation color components
const DS  = 120
const DS1 = 76
const DS2 = 150

' Solution
const MIN_MOVES = 127
const CLKWZ     = 1
const CCLKWS    = 2
const MAX_DELAY = 2000
const DEF_DELAY = 1000
const DELAY_INC = 100

' Mouse Channel
const MCHAN = 2

' Globals
dim outer_track_angles(2, NUM_DISKS)
dim inner_track_angles(NUM_DISKS)
dim disk_colors(NUM_DISKS)
dim disk_pos_colors(NUM_DISKS)
dim xv(NVERTICES), yv(NVERTICES)
dim positions(3*NUM_DISKS+1, 2)
dim disks(NUM_DISKS, 3)
dim slots(NUM_SLOTS)
dim branch_names$(NBRA) = ("LeftOuter", "RightOuter", "Inner")
dim move = 0
dim running = 0
dim start_branch = 3
dim delay = DEF_DELAY
dim automatic = 0

' mouse interface
dim lclick = 0
dim lbusy = 0
dim mouse_x = 0
dim mouse_y = 0

' Main Program
'open "debug.txt" for output as #1
ReadDiskAngles
ReadDiskColors
ReadVertices
InitMouse
InitDiskPositions
NewGame
ShowHelp
HandleEvents
end

' Read vertex coordinates
' combines 12 different segments to produce one polygon
sub ReadVertices
  local i
  for i = 1 to NVERTICES
    read xv(i)
    read yv(i)
  next i
end sub

' Read the disk position angles
sub ReadDiskAngles
  local i
  for i = 1 to NUM_DISKS
    read outer_track_angles(1, i)
  next i
  for i = 1 to NUM_DISKS
    read outer_track_angles(2, i)
  next i
  for i = 1 to NUM_DISKS
    read inner_track_angles(i)
  next i
end sub

' Read the disk colors and the corresponding
' desaturated disk positions.
sub ReadDiskColors
  local i
  for i = 1 to NUM_DISKS
    read disk_colors(i)
  next i
  for i = 1 to NUM_DISKS
    read disk_pos_colors(i)
  next i
end sub

' Initialize Mouse and Cursor
sub InitMouse
  controller mouse open MCHAN, LeftClick
  gui cursor on
  settick 20, UpdateCursor
end sub

' Mouse left click ISR
sub LeftClick
  if not lbusy then
    lclick = 1
    mouse_x = mouse(X)
    mouse_y = mouse(Y)
  end if
end sub

' Make cursor track mouse
sub UpdateCursor
  gui cursor mouse(X), mouse(Y)
end sub

' Initialize disk positions on all 4 track branches
sub InitDiskPositions
  local i, r, k
  r = OUTER_RAD - TRACK_WIDTH\2
  for i = 1 to NUM_DISKS
    angle = outer_track_angles(1, i)
    inc k
    positions(k,1) = CX + r*cos(angle)
    positions(k,2) = CY - r*sin(angle)
  next i
  for i = 1 to NUM_DISKS
    angle = outer_track_angles(2, i)
    inc k
    positions(k, 1) = CX + r*cos(angle)
    positions(k, 2) = CY - r*sin(angle)
  next i
  r = INNER_RAD - TRACK_WIDTH\2
  for i = 1 to NUM_DISKS
    angle = inner_track_angles(i)
    inc k
    positions(k, 1) = CX + r*cos(angle)
    positions(k, 2) = CY - r*sin(angle)
  next i
end sub

' Start a new game by placing the disks on their start positions
sub NewGame
  local i, j, ss
  for i = 1 to NUM_SLOTS
    slots(i) = 0
  next i
  ss = (start_branch-1)*NUM_DISKS + 1
  for i = 1 to NUM_DISKS
    disks(i, DSK_SLOT) = ss+i-1
    disks(i, DSK_INDEX) = i
    disks(i, DSK_SELECTED) = 0
    slots(ss+i-1) = i
  next i
  selected_disk = 0
  selected_branch = 0
  move = 0
  running = 1
  DrawPuzzle
end sub

' Draw the Puzzle
sub DrawPuzzle
  local m$
  page write 1
  cls
  DrawCase
  DrawDiskLandingPositions
  DrawTrack
  DrawDisks
  if move > 0 then ShowMove
  if automatic then
    m$ = "Delay: " + str$(delay) + " msec"
    text 1, 1, m$, "LT"
  end if
  page write 0
  page copy 1 to 0, B
end sub

' Draw the case
sub DrawCase
  local irad = INNER_TRACK_RAD - TRACK_WIDTH
  local y = CY+INNER_TRACK_RAD+55
  local x = mm.hres\2-2
  circle CX, CY, CASE_RAD,,, rgb(black), rgb(gray)
  arc CX, CY, CASE_RAD-10, CASE_RAD-8, 225, 45, rgb(white)
  arc CX, CY, CASE_RAD-10, CASE_RAD-8, 45, 225, rgb(black)
  arc CX, CY, irad-27, irad-25, 225, 45, rgb(white)
  arc CX, CY, irad-27, irad-25, 45, 225, rgb(black)
  line x, y, x, y+100, 2, rgb(white)
  line x, y, x+4, y, 2, rgb(white)
  line x+4, y, x+4, y+100, 2, rgb(black)
  line x, y+100, x+4, y+100, 2, rgb(black)
  box mm.hres\2-80+2, 57+2, 160, 20,, rgb(black)
  box mm.hres\2-80, 57, 160, 20,, rgb(white)
  text mm.hres\2+2, 60+1, "Rudenko's Disk", "CT", 4,, rgb(black), -1
  text mm.hres\2, 60, "Rudenko's Disk", "CT", 4,, rgb(white), -1
end sub

' Draw the landing positions for disks in outer and inner tracks
' The landing colors are somewhat unsaturated so as to better
' distinguish disks from empty landing positions.
sub DrawDiskLandingPositions
  local i, x, y, cx
  r = OUTER_RAD - TRACK_WIDTH\2
  for i = 1 to NUM_SLOTS
    x = positions(i, 1)
    y = positions(i, 2)
    cx = Slot2Disk(i)
    c = disk_pos_colors(cx)
    circle x, y, DISK_OUTER_RAD,,, c, c
    circle x, y, DISK_INNER_RAD,,, c, c
  next i
end sub

' Draw the track
sub DrawTrack
  polygon NVERTICES, xv(), yv(), rgb(black), rgb(black)
end sub  

' Draw the disks in their current positions
sub DrawDisks
  local i, branch, index, selected, x, y, active, offset
  for i = 1 to NUM_DISKS
    slot = disks(i, DSK_SLOT)
    index = disks(i, DSK_INDEX)
    selected = disks(i, DSK_SELECTED)
    x = positions(slot, 1)
    y = positions(slot, 2)
    end select
    active = 0
    DrawDisk i, x, y, selected
  next i
end sub

' Draw a Disk
sub DrawDisk which, x, y, selected
  local cc, oc
  cc = disk_colors(which)
  if selected = 1 then
    oc = rgb(white)
  else if selected = 0 then
    oc = rgb(salmon)
  else if selected = -1 then
    oc = rgb(gray)
    cc = rgb(gray)
  endif
  circle x, y, DISK_OUTER_RAD,,, oc, oc
  circle x, y, DISK_INNER_RAD,,, cc, cc
end sub

' Handle user keyboard inputs
sub HandleEvents
  local z$, cmd
  z$ = INKEY$
  do
    z$ = INKEY$
    cmd = asc(UCASE$(z$))
    select case cmd
      case HOME
        NewGame
      case ESC
        Quit
      case ATK
        AutoSolve
      case PGDWN
        inc start_branch
        if start_branch > 3 then start_branch = 1
        NewGame
      case QUEST
        ShowHelp
      case F1
        ShowHelp2
    end select
    if running then
      if lclick then
        lbusy = 1
        SelectDiskOrTarget
        lclick = 0
        lbusy = 0
      end if
    end if
  loop
end sub

' Quit the program
sub Quit
  settick 0, UpdateCursor
  gui cursor off
  controller mouse close
  cls
  end
end sub

' Move a disk from current location to another if possible.
sub MoveDisk diskno, old_slotno, new_slotno
  local first_diskno, old branch, new_branch
  local path(2*NUM_DISKS+1)
  local sb = Slot2Branch(old_slotno)
  local db = Slot2Branch(new_slotno)
  local fsbs = Branch2FirstSlot(sb)
  local fdbs = Branch2FirstSlot(db)
  if diskno = 0 then exit sub
  if new_slotno = old_slotno then exit sub
  new_branch = Slot2Branch(new_slotno)
  GetClosestDisk new_branch, first_diskno
  if first_diskno > 0 then
    if first_diskno < diskno then exit sub
  end if
  slots(old_slotno) = 0
  disks(diskno, DSK_SLOT) = new_slotno
  slots(new_slotno) = diskno
  inc move
  if IsSolved() then
    DrawPuzzle
    running = 0
    ShowSolution
    exit sub
  end if
  DrawPuzzle
end sub

' Draw the current move number on the screen
sub ShowMove
  local m$
  m$ = "Moves: " + str$(move)
  text mm.hres-1, 1, m$, "RT", 4
end sub

' Select the next disk to be moved or its Destination
' Only the closest disk in a branch can be deselected.
sub SelectDiskOrTarget
  local i, j, dx, dy, slotno, diskno, branch, cdisk, bindex
  local cur_slotno, new_slotno
  local float d
  slotno = 0
  for i = 1 to NUM_SLOTS
    x = positions(i, 1)
    y = positions(i, 2)
    dx = x - mouse_x
    dy = mouse_y - y
    d = sqr(dx*dx+dy*dy)
    if d <= DISK_OUTER_RAD then
      slotno = i
      exit for
    end if
  next i
  if slotno = 0 then exit sub
  branch = Slot2Branch(slotno)
  diskno = slots(slotno)
  if diskno > 0 then
    GetClosestDisk branch, cdisk
    if diskno = cdisk then
      if GetSelectedDisk() = diskno then
        SetSelectedDisk(0)
      else
        SetSelectedDisk(diskno)
      end if
    end if
    exit sub
  end if
  diskno = GetSelectedDisk()
  if diskno = 0 then exit sub
  cur_slotno = disks(diskno, DSK_SLOT)
    GetClosestDisk branch, cdisk
    if cdisk > 0 then
      if cdisk < diskno then exit sub
    end if
    new_slotno = Disk2Slot(branch, diskno)
    MoveDisk diskno, cur_slotno, new_slotno
  end if
end sub

' return the currently-selected disk or 0 if none
function GetSelectedDisk()
  local i, sel
  sel = 0
  for i = 1 to NUM_DISKS
    if disks(i, DSK_SELECTED) then
      sel = i
      exit for
    end if
  next i
  GetSelectedDisk = sel
end function

' Set the specified disk to be selected and unselect all the others
sub SetSelectedDisk diskno
  local i
  for i = 1 to NUM_DISKS
    if i = diskno then
      disks(i, DSK_SELECTED) = 1
    else
      disks(i, DSK_SELECTED) = 0
    end if
  next i
  DrawDisks
end sub

' Given a track branch, find the closest disk in that branch
' and the position it is in. If no disk is found in that branch,
' return 0 for both the disk number
sub GetClosestDisk branch, diskno
  local i, first_slotno, ns
  diskno = 0
  first_slotno = Branch2FirstSlot(branch)
  for i = 1 to NUM_DISKS
    diskno = slots(first_slotno+i-1)
    if diskno > 0 then
      exit for
    end if
  next i  
end sub

' Detect a solved puzzle
function IsSolved()
  local i, solved, sb, eb, slot
  IsSolved = 1
  sb = 0
  for i = 1 to NUM_DISKS
    slot = disks(i, DSK_SLOT)
    eb = Slot2Branch(slot)
    if sb = 0 then
      sb = Slot2Branch(slot)
      if sb = start_branch then
        IsSolved = 0
        exit function
      end if
    end if
    if eb <> 0 then
      if eb <> sb then
        IsSolved = 0
        exit function
      end if
    end if
  next i
end function

' Let user know the puzzle is solved
sub ShowSolution
  local m$
  box 96, 96, 608, 308,, rgb(black), rgb(black)
  box 100, 100, 600, 300,, rgb(cyan), rgb(cyan)
  m$ = "Solved in " + str$(move) + " moves!"
  text 400, 130, m$, "CT", 5,, rgb(black), -1
  if move > 127 then
    text 400, 200, "(But it can be done in fewer moves)", "CT", 4,, rgb(black), -1
  else
    text 400, 200, "Congratulations - minimum moves!!", "CT", 4,, rgb(black), -1
  end if
  text 400, 290, "Press Home key to repeat, Escape key to Quit", "CB",,, rgb(black), -1
  pause 3000
  DrawPuzzle
end sub
    
' function to convert slot number to branch number
function Slot2Branch(slot)
  Slot2Branch = (slot-1)\NUM_DISKS + 1
end function

' function to convert branch number, disk number pair to slot number
function Branch2Slot(branch, disk)
  Branch2Slot = Branch2FirstSlot(branch) + disk - 1
end function

' function to return the first slot number of the specified branch
function Branch2FirstSlot(branch)
  Branch2FirstSlot = (branch-1)*NUM_DISKS + 1
end function

' function to return the disk position index from a slot number
function Slot2Disk(slot)
  Slot2Disk = ((slot-1) mod NUM_DISKS) + 1
end function

' function to return the slot number for the disk position in a branch
function Disk2Slot(branch, diskno)
  local fslot
  fslot = Branch2FirstSlot(branch)
  Disk2Slot = fslot+diskno-1
end function

' Auto-solve the puzzle
sub AutoSolve
  local end_branch, spare, z$
  NewGame
  DrawPuzzle
  select case start_branch
    case 1
      end_branch = 3
      spare = 2
    case 2
      end_branch = 1
      spare = 3
    case 3
      end_branch = 2
      spare = 1
  end select
  automatic = 1
  z$ = INKEY$
  MoveStack NUM_DISKS, start_branch, end_branch, spare
  automatic = 0
end sub

' Move a stack of 'num' Disks from branch 'source' to branch 'dest' (recursive!)
sub MoveStack num, source, dest, spare
  local sslot, dslot, t0, et
  t0 = timer  
  if num = 1 then
    sslot = Branch2Slot(source, num)
    dslot = Branch2Slot(dest, num)
    et = timer-t0
    if et < delay then pause max(delay-et, 0)
    MoveDisk num, sslot, dslot
    pause max(delay-et, 0)
    GetDelayUpdate
  else
    MoveStack num-1, source, spare, dest
    sslot = Branch2Slot(source, num)
    dslot = Branch2Slot(dest, num)
    et = timer-t0
    if et < delay then pause max(delay-et, 0)
    MoveDisk num, sslot, dslot
    GetDelayUpdate
    MoveStack num-1, spare, dest, source
  end if
  et = timer-t0
  if et < delay then pause max(delay-et, 0)
end sub

'Get user input to change the delay interval while auto-solving
sub GetDelayUpdate
  local z$, cmd
  z$ = INKEY$
  if z$ <> "" then
    cmd = asc(UCASE$(z$))
    select case cmd
      case PLUS
        inc delay, DELAY_INC
        if delay > MAX_DELAY then delay = MAX_DELAY
      case MINUS
        inc delay, -DELAY_INC
        if delay < 0 then delay = 0
    end select
  end if
end sub
  
' Show the Help Screen
sub ShowHelp
  local z$
  cls
  gui cursor hide
  load png "Rudenko.png", 590, 400
  text mm.hres\2, 10, "Rudenko's Disk", "CT", 4,, rgb(green)
  print @(0, 40)
  print "Rudenko's Disk (picture below) is a plastic puzzle where your task is to move the"
  print "7 colored beads from the inner curved track where they start to one of the other"
  print "2 outer curved tracks that are marked with the beads' colors. To complete the"
  print "puzzle, all the beads must end up on the same outer track."
  print ""
  print "This program is an emulation of Rudenko's Disk, and your task is the same as with"
  print "the physical puzzle."
  print ""
  print "To move a disk, select it with the mouse. The selected disk will show a white border"
  print "to let you know it is selected. Then click somewhere on the track you want to move"
  print "it to. It doesn't matter exactly where in the track you click, because the disk"
  print "will be automatically sent to the location whose color matches the disk color."
  print ""
  print "Each time you move a disk, you have a choice of 2 locations to move it to. 
  print "You will quickly find out that there cannot be any other disks that intervene"
  print "between the disk you want to move and the location you want to send it to."
  print "This means that if multiple disks are on the same track, only the smallest one"
  print "can be selected and moved."
  print ""
  print "But in spite of these limitations, the puzzle can be solved! If you get frustrated,"
  print "Press the '@' key on the keyboard to see an automated solution in the mininum"
  print "number of moves. Pressing '+' or '-' changes speed of moves."
  print ""
  print "This puzzle is one of a class of 'N-ary' puzzles. Puzzles in this class require an"
  print "exponential number of moves depending on how many items need to be moved. So for"
  print "this puzzle, moving 7 disks takes about twice as many moves as it would take if there"
  print "where only 6 disks, and 6 disks take about twice as many moves as 5 disks, etc."
  print "Other N-ary puzzles include Towers of Hanoi, the Chinese Ring puzzle, and the"
  print "new 3D-printed 'Zigguflat' puzzle."
  print "The puzzle will inform you when it has been solved."
  print ""
  print "Note: You can choose which track the beads start on:"
  print "inner (default), outer left, or outer right,"
  print "Press the PageDown key to cycle through start positions."
  print ""
  print "To learn more about N-ary puzzles, press the F1 key,"
  text mm.hres\2, mm.vres-1, "Press Any Key to Continue", "CB"
  z$ = INKEY$
  do
    z$ = INKEY$
  loop until z$ <> ""
  if asc(z$) = F1 then ShowHelp2
  cls
  gui cursor show
  DrawPuzzle
end sub

' Show additional Help Info
sub ShowHelp2
  local z$
  cls
  gui cursor hide
  text mm.hres\2, 1, "More on N-ary Puzzles", "CT", 4,, rgb(green)
  print @(0, 30)
  print "N-ary puzzles are puzzles that require exponentially more moves to complete as the"
  print "number of pieces and the number of choices increase. The classic examples of this"
  print "kind of puzzle are the Chinese rings and the Towers of Hanoi. Both of these"
  print "puzzles are 2-ary (binary), because at each move there are exactly 2 choices of where to"
  print "move the next piece."
  print ""
  print "Other puzzles may be 3-ary, 4-ary, and so on, if they have 3, 4 or more choices at"
  print "each move. The number of moves grows even faster per piece than with the binary puzzles,"
  print "and may grow to thousands of moves for just a few pieces."
  print ""
  print "Below are photos of the 2-ary (binary) Chinese Rings puzzle and two 3-ary puzzles:"
  print "the Constantin Sliding puzzle and the Zigguflat puzzle by Oskar Van Deventer."
  text 100, 248, "Chinese Rings", "LB"
  load png "ChineseRings.png", 50, 250
  text 350, 298, "Constantin", "LB"
  load png "Nary4x4.png", 300, 300
  text 650, 398, "Zigguflat", "LB"
  load png "Zigguflat.png", 600, 400
  text mm.hres\2, mm.vres-1, "Press Any Key to Continue", "CB"
  z$ = INKEY$
  do
    z$ = INKEY$
  loop until z$ <> ""
  cls
  gui cursor show
  DrawPuzzle
end sub

' Disk Angles for Outer Track
data 117, 138, 159, 180, 201, 222, 243 ' left
data 64,  43,  22,  1,   -20, -41, -62 ' right

' Disk Angles for Inner Track
data 405, 360, 315, 270, 225, 180, 135
'data 135, 180, 225, 270, 315, 360, 405

' Disk Colors
data rgb(red), rgb(255, 127, 0), rgb(yellow), rgb(green), rgb(cyan), rgb(blue), rgb(magenta)

' Disk Position Colors (0.25 desaturated)
data rgb(255, DS1, DS1), rgb(255, 127, DS), rgb(255, 255, DS), rgb(DS, 255, DS)
data rgb(DS2, 255, 255), rgb(DS2, DS2, 255), rgb(255, DS, 255)
    
' Data points for Outer Track outer edge left (26 points)
data 299, 110, 283, 120, 268, 131, 253, 143, 240, 156, 228, 171, 218, 186, 208, 202
data 201, 219, 194, 237, 190, 255, 187, 274, 185, 292, 185, 311, 187, 330, 191, 348
data 196, 366, 202, 384, 210, 401, 220, 417, 231, 432, 243, 447, 256, 460, 271, 472
data 286, 482, 302, 492

' Outer Left End (5 points)
data 306, 493, 312, 492, 317, 489, 320, 483, 319, 477

' Data points for Outer Track inner edge left (26 points)
data 314, 469, 299, 461
data 286, 452, 273, 441, 261, 430, 250, 417, 241, 403, 232, 389, 225, 374, 219, 359
data 215, 343, 212, 326, 210, 310, 210, 293, 211, 277, 214, 260, 218, 244, 224, 229
data 231, 214, 239, 199, 248, 186, 259, 173, 270, 161, 283, 150, 297, 141, 311, 132

' Data points for the transition curve left edge (17 points)
data 318, 127, 330, 124, 340, 126, 350, 130, 360, 136, 370, 142, 378, 149, 383, 160
data 384, 174, 385, 183, 387, 195, 390, 200, 395, 205, 400, 207, 405, 209, 410, 211
data 415, 213

' Data points for Inner Track inner edge (31 points)
data 419, 217, 433, 222, 446, 229, 458, 238, 468, 249, 476, 261, 481, 275
data 484, 290, 485, 304, 483, 319, 478, 333, 471, 346, 462, 358, 451, 368, 439, 376
data 425, 381, 410, 384, 396, 385, 381, 383, 367, 378, 354, 371, 342, 362, 332, 351
data 324, 339, 319, 325, 316, 319, 315, 296, 317, 281, 322, 267, 329, 254, 338, 242

' Inner Track End (5 points)
data 343, 233, 342, 227, 339, 222, 333, 219, 327, 220

' Data points for Inner Track outer edge (31 points)
data 320, 225, 308, 240, 299, 257, 293, 275, 290, 294, 291, 313, 295, 332, 302, 350
data 312, 366, 325, 380, 340, 392, 357, 401, 375, 407, 394, 410, 413, 409, 432, 405
data 450, 398, 466, 388, 480, 375, 492, 360, 501, 343, 507, 325, 510, 306, 509, 287
data 505, 268, 498, 250, 488, 234, 475, 220, 460, 208, 443, 199, 425, 193

' Data points for the transition curve right edge (12 points)
data 416, 185, 413, 180, 410, 170, 412, 160, 418, 150, 429, 140
data 445, 130, 454, 127, 460, 126, 470, 126, 480, 128, 490, 132

' Data points for the outer track inner edge right (26 points)
data 489, 132, 501, 139, 514, 148, 527, 159, 539, 170, 550, 183, 559, 197
data 568, 211, 575, 226, 581, 241, 585, 257, 588, 274, 590, 290, 590, 307, 589, 323
data 586, 340, 582, 356, 576, 371, 569, 386, 561, 401, 552, 414, 541, 427, 530, 439
data 517, 450, 503, 459, 489, 468

' Outer Right End (5 points)
data 484, 475, 483, 481, 486, 487, 491, 490, 497, 491

' Data points for the outer track outer edge right (21 points)
'data 484, 498
data 547, 457, 560, 444, 572, 429, 582, 414, 592, 398, 599, 381, 606, 363, 610, 345
data 613, 326, 615, 308, 615, 289, 613, 270, 609, 252, 604, 234, 598, 216, 590, 199
data 580, 183, 569, 168, 557, 153, 544, 140, 529, 128, 514, 118, 501, 110

' Data points for transition curve top (14 points)
data 495, 107, 485, 104, 475, 102, 465, 101, 455, 101, 445, 102, 421, 114
data 397, 126, 372, 114, 348, 102, 338, 101, 328, 101, 318, 102, 308, 104

